unit odf2pdflibrary;

{ *********************************************************
  ODF2PDF library
  Copyright (c) 2019, 2020 Erick Engelke

  Take an ODF (Open Document Formated) file, do a mailmerge of
  data replacement and create a PDF.  Optionally Save that PDF,
  or transmit it over the web.

  Dependancy: SynZip and SynPDF copyright their respecitive owners
  ********************************************************* }
interface

uses
  System.SysUtils, classes, synpdf, synzip, vcl.graphics, System.zip;

{$INCLUDE SYNOPSE.INC}
{ //$DEFINE USE_PDFSECURITY }
{ //$DEFINE DEBUG1 }  // lots of debugging info gets printed
{ //$DEFINE GDIPLUSDRAW }
{$DEFINE NOLINES }

// disable lines around tables, not fully ready yet
type
  eexml = (eeTop, eeComment, eeNodeEntry, eeNodeEntrySub, eeNodeType,
    eeNodeExit, eeNodeAttribName, eeNodeAttribType, eeNodeAttribData, eeData);

  TEEAttrib = class(TObject)
  public
    name: string;
    nametype: string;
    data: string;
  end;

  TEENode = class(TObject)
  public
    parent: TEENode;
    parentopen: boolean;
    stylename: string;
    nodetype: string;
    nodesubtype: string;
    // subtypes: TStringList;
    name: string;
    attributes: TStringList;
    data: string;
    subnodes: TStringList;

    left: integer; // for columns unused presently
    width: integer;

    constructor Create(parentnode: TEENode);
    destructor Destroy; override;
  end;

  TEETableElement = class(TObject)
  public
    name: string;
    safetysig: uint32;
    columnsspanned: integer;
    pxleft: integer;
    pxright: integer;
    pxtop: integer;
    pxbottom: integer;
    width: integer;
    effectivewidth: integer; // handle colsspan
    savetop: integer;

    toppage: integer;
    rightmost: integer; // how far right did we print, for followup
  end;

  TEETableColumn = class(TEETableElement)
    //
  end;

  TEETableRow = class(TEETableElement)
    //
  end;

  TEETable = class(TObject)
  public
    safety: uint32;
    isequalized: boolean;
    pagestart: integer; // first page
    pxtop: integer; // first top
    columns: TStringList;
    // rows: TStringList;
    width: integer;
    cellindex: integer;
    currow: TEETableRow;
    curcol: TEETableColumn;
    curcell: TEETableElement;
  end;

  TEEDrawLater = class(TObject)
  public
    page: integer;
    text: ansistring;
    bold: boolean;
    fontsize: integer;
    x, y: integer;
    x2, y2: integer; // for line draw
    col: TCOlor;
  end;

type
  TEEStyle = class(TObject)
  public
    name: string;
    family: string;
    columnsspanned: integer;
    dobold: boolean;
    fontsize: integer;
    eenode: TEENode;
    destructor Destroy; override;
  end;

  TEEPDFReplace = function(input: string): string;

  TEEPDFDocument = class(TPDFDocumentGDI)
  private
    //
    // procedure SetOrientation(const Value: TPrinterOrientation);
    // procedure SetDefaultPaperSize(const Value: TPDFPaperSize);
    curx, cury: integer; // points on the page
    curpage: integer;
    tempcurpage: integer;

    ourEENode: TEENode;
    curtable: TEETable; // active
    // lpage: TPdfPage;

    drawlatertable: TStringList;
    lastpage, lasty: integer;
    maxlineheight: integer; // set to maximum so next line will be safe
    insidereplacements: boolean;
    insideSpanReplace : string;
    // function GetFirstNode: TEENode;
    // function GetNextNode: TEENode;
    procedure eepdfprocessone(x: TEENode);
    procedure QueueDrawLater(lpage: integer; x, y: integer; text: string;
      usebold: boolean; fontsize: integer);
    procedure QueueLineDraw(lpage: integer; x, y: integer; x2, y2: integer;
      col: TCOlor);
    procedure NowDrawQueue;
    procedure nextline(optionalfontsize: integer);
    // procedure reequalizecolumns;
    procedure FixCurrentCellWidth(colsspanned: integer);

    procedure DrawTableLines;
    procedure SetDefaultPaperSize(x: TPdfPaperSize);
    function processreplacements(s: string): string;
  published
    property PaperSize: TPdfPaperSize write SetDefaultPaperSize;
  public
    // MarginLeft, MarginTop, MarginRight, MarginBottom: Double;

    leftmargin: integer;
    rightmargin: integer;
    topmargin, bottommargin: integer;
//    ScaleToFit: boolean;
    DrawPageNumber: boolean;
    DrawPageNumberText: string;
    top: TEENode;
    eestyles: TStringList;
    curstyle: string;
    somewords: TStringList;
    replacements: TStringList;
    replacementserror: TStringList;
    lineheight: integer;
    fontheight: integer;
    fontsize: integer;
    fixedfontsize: boolean;  // force system to just use one font size
    font: string;

    function AddStyle(name, family: string; ee: TEENode): TEEStyle;
    function GetSubAttrSub(ee: TEENode; name, nametype: string): string;
    function GetSubAttr(stylename, name, nametype: string): string;
    function GetStyle(stylename: string): TEEStyle;
    // procedure eeprocessone(x: TEENode);
    procedure dumpee(indent: string; x: TEENode);
    procedure LoadFromXMLFile(filename: string);
    function Inches2Points(AInches: Double): Single;
    function Centimeters2Points(ACentimeters: Double): Single;
    function ExtractPixelString(s: string): integer;

    procedure NewTable;
    procedure NewRow(namex: string);
    procedure NewCol(namex: string);
    procedure NewCell(namex: string);
    procedure NewPage;

    constructor Create(AUSEOUTLINES: boolean = false; ACodePage: integer = 0;
      APDFA1: boolean = false{$IFDEF USE_PDFSECURITY};
      AEncryption: TPdfEncryption = nil{$ENDIF});

    destructor Destroy; override;
    function GeneratePDF: string;
    procedure LoadFromODT(filename: string);

  end;

  // --------------------------------------------------

implementation

procedure TEEPDFDocument.SetDefaultPaperSize(x: TPdfPaperSize);
begin
  inherited SetDefaultPaperSize(x);
end;

constructor TEENode.Create(parentnode: TEENode);
begin
  inherited Create;
  parent := parentnode;

  attributes := TStringList.Create;
  subnodes := TStringList.Create;
end;

destructor TEENode.Destroy;
var
  i: integer;
  a: TEEAttrib;
  x: TEENode;
begin
  for i := 0 to subnodes.Count - 1 do
  begin
    x := subnodes.Objects[i] as TEENode;
    if Assigned(x) then
      x.Free;
  end;
  subnodes.Free;
  for i := 0 to attributes.Count - 1 do
  begin
    a := attributes.Objects[i] as TEEAttrib;
    if Assigned(a) then
      a.Free;
  end;
  attributes.Free;
  // subtypes.Free;

end;

{
  procedure TEEPDFDocument.reequalizecolumns;
  var
  count, i, j: integer;
  c, d: TEETableElement;
  begin
  if curtable.columns.Count = 0 then exit;
  for i := 0 to curtable.columns.Count -1 do
  begin
  c := TEETableElement(curtable.columns.Objects[i]);
  j := c.columnsspanned; // 1 or more
  if (j < 1) or (j+i-1 >= curtable.columns.Count) then
  raise Exception.Create('Invalid column-span');

  // if it is more than one column wide, reflect that
  d := TEETableElement(curtable.columns.Objects[i+j-1]);
  if j > 1  then

  writeln( format('%d %d %d ',[ c.pxleft, c.pxright, d.pxright] ));

  c.width := d.pxright - c.pxleft;
  c.pxright := d.pxright;
  end;
  end;
}
procedure TEEPDFDocument.FixCurrentCellWidth(colsspanned: integer);
var
  Count, i, j: integer;
  c, d: TEETableColumn;
begin

  Count := curtable.columns.Count;
  i := curtable.cellindex;

  if Count = 0 then
    exit;

  if i = Count - 1 then
  begin
    curtable.curcell.pxright := rightmargin;
    exit;
  end;

  if colsspanned = 0 then
    exit; // save time

  if (colsspanned < 1) or (colsspanned + i - 1 >= curtable.columns.Count) then
    raise Exception.Create('Invalid column-span');

  // if it is more than one column wide, reflect that
  d := TEETableColumn(curtable.columns.Objects[i + colsspanned - 1]);

  curtable.curcell.width := d.pxright - curtable.curcol.pxleft;
  curtable.curcell.pxright := d.pxright;
end;

procedure TEEPDFDocument.QueueDrawLater(lpage: integer; x, y: integer;
  text: string; usebold: boolean; fontsize: integer);
var
  temp: TEEDrawLater;
begin
  if text <> '' then
  begin
    temp := TEEDrawLater.Create;
    temp.page := lpage;
    temp.x := x;
    temp.y := y;
    temp.text := text;
    temp.bold := usebold;
    temp.fontsize := fontsize;
    drawlatertable.AddObject('', temp);
    if lastpage < lpage then
    begin
      lasty := y;
      lastpage := lpage;
    end
    else if (lasty > y) or (lasty = 0) then
      lasty := y;

  end;
end;

procedure TEEPDFDocument.QueueLineDraw(lpage: integer; x, y: integer;
  x2, y2: integer; col: TCOlor);
var
  temp: TEEDrawLater;
begin
  temp := TEEDrawLater.Create;
  temp.page := lpage;
  temp.x := x;
  temp.y := y;
  temp.y2 := y2;
  temp.x2 := x2;
  temp.col := col;
  drawlatertable.AddObject('', temp);

  if x2 > 0 then

  begin
    if lastpage < lpage then
    begin
      lastpage := lpage;
      lasty := y;
    end
    else if lasty < y then
  end
end;

procedure TEEPDFDocument.NewPage;
begin
  addpage;
  with canvas do
  begin
    SetFont('Arial', 9, []);
    SetRGBStrokeColor(clBlack);
    fontheight := 9;
    lineheight := Trunc(fontheight + 3);
  end;
end;

procedure TEEPDFDocument.NowDrawQueue;
var
  page, j: integer;
  temp: TEEDrawLater;
  tempfontsize: integer;
begin
  for page := 0 to lastpage do
  begin
    // fpage :=
    if page > 0 then
      NewPage;
    for j := 0 to drawlatertable.Count - 1 do
    begin
      temp := drawlatertable.Objects[j] as TEEDrawLater;
      if temp.page = page then
      begin
        if temp.x2 > 0 then
        begin
          with canvas do
          begin

            // pen.width := MulDiv(fDefaultLineWidth, Self.Font.size, 8);
            // if fsBold in Self.Font.style then
            // Pen.width := Pen.width + 1;
{$IFNDEF NOLINES}
            SetRGBStrokeColor(temp.col);
            SetLineWidth(1);
            MoveTo(temp.x, temp.y);
            LineTo(temp.x2, temp.y2);
            stroke;
{$ENDIF}
          end;

        end
        else
        begin
          tempfontsize := fontsize;
          if not fixedfontsize then
          begin
            if temp.fontsize > 0 then
            begin
              tempfontsize := temp.fontsize;
            end;
          end;

          if temp.bold then
            canvas.SetFont(font, tempfontsize, [pfsbold])
          else
            canvas.SetFont(font, tempfontsize, []);

          canvas.textout(temp.x, temp.y - tempfontsize + 3 { lineheight } ,
            temp.text);
        end;
      end;
    end;
  end;
end;

procedure TEEPDFDocument.nextline(optionalfontsize: integer);
begin
  if optionalfontsize = 0 then
    optionalfontsize := maxlineheight;

  if optionalfontsize = 0 then

    optionalfontsize := lineheight
  else
    inc(optionalfontsize, 3);

  dec(cury, optionalfontsize);

  curx := curtable.curcell.pxleft;
  if cury < bottommargin then
  begin
    cury := topmargin;
    inc(tempcurpage);
  end;
  // return to default
  maxlineheight := lineheight;
end;

function ParseEEXML(s: string): TEENode;
var
  curnode, newnode, textnode: TEENode;
  topnode: TEENode;
  ee: eexml;
  ch, nextch: char;
  i: integer;
  len: integer;
  attrindex: integer;
  curattr: TEEAttrib;
  temp: string;
  inquotes: boolean;
  lastwasopen: boolean;
begin
  i := 1;
  len := Length(s);

  ee := eeData;
  topnode := TEENode.Create(Nil);
  curnode := topnode;
  result := topnode;
  lastwasopen := false;

  i := 1;
  while i <= len do
  begin
    ch := s[i];
    if i < len then
      nextch := s[i + 1]
    else
      nextch := ' ';
    if (ee = eeData) and (ch <> '<') then
    begin
      if not Assigned(textnode) then
        raise Exception.Create('Starts with text');
      textnode.data := textnode.data + ch;
    end
    else

      case ch of
        '?': // skip comment
          begin
            temp := '';
            repeat
              ch := s[i];
              inc(i);
              temp := temp + ch;

            until (i = len) or (ch = '>');
{$IFDEF DEBUG1}
            writeln('sub : ' + temp);
{$ENDIF}
          end;

        '<':
          begin
            if nextch = '/' then
            begin

              ee := eeNodeExit;
            end
            else
            begin
              newnode := TEENode.Create(curnode); // begin a node
              // is is successor (not inherited (closed)), or descendant (inherited/open)
              newnode.parentopen := lastwasopen;

              lastwasopen := True;
              curnode.subnodes.AddObject('1', newnode);
              curnode := newnode;

              ee := eeNodeEntry;
            end;
          end;
        '/':
          begin
            ee := eeNodeExit;
            lastwasopen := false;
          end;
        '>': // close node
          begin

            if ee = eeNodeExit then
            begin
              curnode := curnode.parent;
              if curnode = Nil then
                exit;

            end
            else
            begin
              // intermediate data
            end;

            ee := eeData;
            textnode := TEENode.Create(curnode);
            curnode.subnodes.AddObject('', textnode);
          end;
        'A' .. 'Z', 'a' .. 'z', '0' .. '9', '-', '.':
          case ee of
            eeData:
              curnode.data := curnode.data + ch;
            eeNodeEntry:
              curnode.nodetype := curnode.nodetype + ch;
            eeNodeEntrySub:
              curnode.nodesubtype := curnode.nodesubtype + ch;
            eeNodeExit: { skip }
              ;
            eeNodeAttribName:
              curattr.name := curattr.name + ch;
            eeNodeAttribType:
              curattr.nametype := curattr.nametype + ch;
          else
            Raise Exception.Create(format('ERROR: unnown state %d', [ord(ee)]));
          end;

        ' ', #$0a, #$0d:
          case ee of
            eeNodeEntry, eeNodeExit, eeNodeEntrySub, eeNodeAttribName,
              eeNodeAttribType, eeNodeAttribData:
              begin
                ee := eeNodeAttribName;
                curattr := TEEAttrib.Create;
                attrindex := curnode.attributes.AddObject('', curattr);
              end;
          else
            raise Exception.Create(format('unexpected space %', [ord(ee)]));

          end;

        ':':
          case ee of
            eeNodeEntry:
              ee := eeNodeEntrySub;
            eeNodeEntrySub:
              raise Exception.Create('Error: bad xml, too many sub types ' +
                curnode.nodetype + ':' + curnode.nodesubtype);
            eeNodeExit:
              ee := ee;
            eeNodeAttribName:
              ee := eeNodeAttribType;
          end;
        '=':
          case ee of
            eeNodeAttribName, eeNodeAttribType:
              begin
                inc(i);
                inquotes := s[i] = '"';
                if inquotes then

                  inc(i);
                ch := s[i];

                while (i < len) and (ch <> '"') do
                begin
                  curattr.data := curattr.data + ch;
                  inc(i);
                  ch := s[i];
                  if not inquotes then
                    if ch = '<' then
                      break;

                end;
              end;
          end;
      else
        begin
          raise Exception.Create(format('Illegal char %c', [ch]));
        end;
      end;
    inc(i);
  end;

end;

destructor TEEStyle.Destroy;
begin
  inherited;
end;

function TEEPDFDocument.AddStyle(name, family: string; ee: TEENode): TEEStyle;
var
  ees: TEEStyle;
  i: integer;
begin
  ees := TEEStyle.Create;
  ees.name := name;
  ees.family := family;
  ees.eenode := ee;

{$IFDEF DEBUG1}
  writeln('ADDING STYLE ', name, ' family ', family, '  ', ee.nodetype);
{$ENDIF}
  if eestyles.Find(name, i) then
  BEGIN
    // writeln('duplicate style: ', name);
    ees.Free;
    ees := eestyles.Objects[i] as TEEStyle;
  end
  else
    eestyles.AddObject(name, ees);
  result := ees;
end;

function TEEPDFDocument.GetStyle(stylename: string): TEEStyle;
var
  i: integer;
  a: TEEAttrib;
begin
  if not eestyles.Find(stylename, i) then
    result := Nil
  else
    result := eestyles.Objects[i] as TEEStyle;
end;

function TEEPDFDocument.GetSubAttrSub(ee: TEENode;
  name, nametype: string): string;
var
  i: integer;
  a: TEEAttrib;

begin
  for i := 0 to ee.attributes.Count - 1 do
  begin
    a := ee.attributes.Objects[i] as TEEAttrib;
    if SameText(a.name, name) and SameText(a.nametype, nametype) then
    begin
      result := a.data;
      exit;
    end;

  end;

end;

function TEEPDFDocument.GetSubAttr(stylename, name, nametype: string): string;

var
  i: integer;
  ees: TEEStyle;
  a: TEEAttrib;
  ee: TEENode;
begin

  result := '';
  if eestyles.Find(stylename, i) then
  begin
    ees := eestyles.Objects[i] as TEEStyle;
    result := GetSubAttrSub(ees.eenode, name, nametype);
    if result <> '' then
      exit;
    // didn't find yet, look ddept
    for i := 0 to ees.eenode.subnodes.Count - 1 do
    begin
      ee := ees.eenode.subnodes.Objects[i] as TEENode;

      // writeln('ee : ', ee.nodetype, ':', ee.nodesubtype, ' = ', ee.data);
      result := GetSubAttrSub(ee, name, nametype);
      if result <> '' then
        exit;
    end;
  end
  else
  begin
    // raise Exception.Create('missing style ' + stylename);
  end;
end;

function TEEPDFDocument.ExtractPixelString(s: string): integer;
var
  i: integer;
  d: Double;
begin
  for i := Length(s) downto 1 do
  begin
    if s[i] < 'A' then
    begin
      d := StrToFloatDef(COpy(s, 1, i), 0);
      s := COpy(s, i + 1);
      result := Trunc(d);
      if SameText(s, 'IN') then
        result := Trunc(Inches2Points(d));
      if SameText(s, 'CM') then
        result := Trunc(Centimeters2Points(d));
      exit;
    end;
  end;

end;

(*
  procedure TEEPDFDocument.eeprocessone(x: TEENode);
  var
  i: integer;
  a: TEEAttrib;
  sub: TEENode;
  stylename, stylefamily: string;
  begin
  try
  { //$DEFINE VERBOSEDEBUG1 }
  {$IFDEF VERBOSEDEBUG1}
  writeln(x.nodetype, '  --- ');
  {$ENDIF}
  if (x.nodetype = '') and (x.nodesubtype = '') then
  if x.data <> '' then
  writeln(x.data);

  if SameText(x.nodetype, 'style') { and SameText(x.nodesubtype, 'style') }
  then
  begin
  stylename := '';
  stylefamily := '';
  for i := 0 to x.attributes.Count - 1 do
  begin
  a := TEEAttrib(x.attributes.Objects[i]);
  if SameText(a.name, 'STYLE') then
  begin
  if SameText(a.nametype, 'NAME') then
  stylename := a.data;

  if SameText(a.nametype, 'FAMILY') then
  stylefamily := a.data;
  // if SameText(a.nametype, 'FONT') ;

  end;

  end;
  if (stylename <> '') and (stylefamily <> '') then
  AddStyle(stylename, stylefamily, x);

  end

  else if SameText(x.nodetype, 'TEXT') then
  begin

  // look at attribute
  for i := 0 to x.attributes.Count - 1 do
  begin
  a := x.attributes.Objects[i] as TEEAttrib;
  if SameText(a.name, 'TEXT') and SameText(a.nametype, 'STYLE-NAME') then
  begin
  curstyle := a.data;
  writeln('STYLE : ', a.data);
  end;
  end;

  {$IFDEF VERBOSEDEBUG1}
  write('Text: '); // writeln( a.data)
  {$ENDIF}
  if SameText(x.nodesubtype, 'TAB') then
  writeln('TAB');
  end
  else if SameText(x.nodetype, 'TABLE') then
  begin
  if SameText(x.nodesubtype, 'TABLE') then
  begin

  curstyle := GetSubAttrSub(x, 'TABLE', 'STYLE-NAME');
  if curstyle = '' then
  raise Exception.Create('Table missing style');

  writeln('TABLETABLE!');
  writeln('   - width ', GetSubAttr(curstyle, 'STYLE', 'WIDTH'));
  end

  else if SameText(x.nodesubtype, 'TABLE-COLUMN') then
  begin
  curstyle := GetSubAttrSub(x, 'TABLE', 'STYLE-NAME');
  if curstyle = '' then
  raise Exception.Create('Table missing style');
  writeln('TABLE COLUMN!');

  end
  else if SameText(x.nodesubtype, 'TABLE-ROW') then
  begin
  curstyle := GetSubAttrSub(x, 'TABLE', 'STYLE-NAME');
  if curstyle = '' then
  raise Exception.Create('Table missing style');
  writeln('TABLE ROW!');
  end
  else if SameText(x.nodesubtype, 'TABLE-CELL') then
  begin
  curstyle := GetSubAttrSub(x, 'TABLE', 'STYLE-NAME');
  if curstyle = '' then
  raise Exception.Create('Table missing style');
  writeln('TABLE CELL!');
  end
  else
  write('Table: ', x.nodesubtype); // writeln(a.data);

  end
  else if x.nodetype <> '' then
  writeln('UNKNOWN : ', x.nodetype);
  except
  writeln('ERROR: processing ', x.nodetype);
  raise;
  end;

  for i := 0 to x.subnodes.Count - 1 do
  begin
  sub := x.subnodes.Objects[i] as TEENode;
  if not Assigned(sub) then
  raise Exception.Create('Missing node');

  eeprocessone(sub);
  end;

  end;
*)
procedure TEEPDFDocument.dumpee(indent: string; x: TEENode);
var
  i: integer;
  a: TEEAttrib;
  sub: TEENode;
begin
{$IFDEF DEBUG1}
  writeln(indent, 'TYPE: ', x.nodetype, ':', x.nodesubtype, ' = "',
    x.data, '"');
{$ENDIF}
  for i := 0 to x.attributes.Count - 1 do
  begin
    a := x.attributes.Objects[i] as TEEAttrib;
{$IFDEF DEBUG1}
    writeln(indent, '  attr', i, ' ', a.name, ':', a.nametype, '= "',
      a.data, '"');
{$ENDIF}
  end;
  for i := 0 to x.subnodes.Count - 1 do
  begin
    sub := x.subnodes.Objects[i] as TEENode;
    dumpee(indent + '   ', sub);
  end;
{$IFDEF DEBUG1}
  writeln(indent, 'DATA:"', x.data, '"');
{$ENDIF}
end;

procedure TEEPDFDocument.LoadFromXMLFile(filename: string);
var
  sl: TStringList;
  s: string;
begin
  sl := TStringList.Create;
  sl.LoadFromFile(filename);
  s := sl.text;
  sl.Free;
  top := ParseEEXML(s);

end;

constructor TEEPDFDocument.Create(AUSEOUTLINES: boolean = false;
  ACodePage: integer = 0; APDFA1: boolean = false{$IFDEF USE_PDFSECURITY};
  AEncryption: TPdfEncryption = nil{$ENDIF});

begin
  inherited Create(AUSEOUTLINES, ACodePage, APDFA1{$IFDEF USE_PDFSECURITY},
    AEncryption{$ENDIF});

  // set some defaults
  fontsize := 9;


  eestyles := TStringList.Create;
  eestyles.Sorted := True;
  somewords := TStringList.Create;
  drawlatertable := TStringList.Create;
  replacements := TStringList.Create;
  replacementserror := TStringList.Create;
end;

destructor TEEPDFDocument.Destroy;
var
  i: integer;
begin
  replacements.Free;
  replacementserror.Free;

  if Assigned(eestyles) then
  begin

    for i := 0 to eestyles.Count - 1 do
      eestyles.Objects[i].Free;
    eestyles.Free;
  end;

  if Assigned(drawlatertable) then
  begin

    for i := 0 to drawlatertable.Count - 1 do
      drawlatertable.Objects[i].Free;
    drawlatertable.Free;

  end;
  if Assigned(curtable) then
  begin

    for i := 0 to curtable.columns.Count - 1 do
    begin
      curtable.columns.Objects[i].Free;
    end;
    curtable.columns.Free;
    if Assigned(curtable.currow) then
      curtable.currow.Free;
    if curtable.safety <> $1234FEED then
      raise Exception.Create('curtable safety sig');

    if Assigned(curtable.curcell) then
      curtable.curcell.Free;

    curtable.Free;
  end;

  somewords.Free;
  top.Free;

  // destroyall
  inherited;
end;

const
  CPointsPerInch = 72;
  CCentimetersPerInch = 2.54;

function TEEPDFDocument.Inches2Points(AInches: Double): Single;
begin
  result := AInches * CPointsPerInch;
end;

function TEEPDFDocument.Centimeters2Points(ACentimeters: Double): Single;
begin
  result := ACentimeters / CCentimetersPerInch * CPointsPerInch;
end;

{
  function TEEPDFDocument.GetFirstNode: TEENode;
  begin
  ourEENode := top;
  result := ourEENode;
  end;

  function TEEPDFDocument.GetNextNode: TEENode;
  begin
  ourEENode := Nil;
  result := ourEENode;

  end;
}
procedure TEEPDFDocument.NewCell(namex: string);
begin
  if Assigned(curtable.curcell) then
    curtable.curcell.Free;

  curtable.curcell := TEETableElement.Create;
  with curtable.curcell do
  begin
    pxleft := leftmargin;
    pxright := rightmargin;
    pxtop := topmargin;
    pxbottom := topmargin; // bottommargin;
    name := namex;
  end;

end;

procedure TEEPDFDocument.NewCol(namex: string);
begin

  if curtable.columns.IndexOf(namex) > -1 then
  begin
    raise Exception.Create('Reusing column ' + namex);
  end;

  curtable.curcol := TEETableColumn.Create;
  curtable.columns.AddObject(namex, curtable.curcol);

  with curtable.curcol do
  begin
    safetysig := $12344321;
    pxleft := leftmargin;
    pxright := rightmargin;
    pxtop := -1;
    pxbottom := -1;
    name := namex;
  end;

end;

procedure TEEPDFDocument.NewRow(namex: string);
begin
  if Assigned(curtable.currow) and false then
  begin
    if curtable.currow.safetysig <> 55554444 then
      raise Exception.Create('Signature error on row');
    curtable.currow.Free;
  end;

  curtable.currow := TEETableRow.Create;
  with curtable.currow do
  begin
    pxleft := -1;
    pxright := -1;
    pxtop := topmargin;
    pxbottom := topmargin; // bottommargin;
    name := namex;
    safetysig := 55554444;
  end;

end;

procedure TEEPDFDocument.NewTable;
var
  i: integer;
begin
{$IFDEF DEBUG1}
  writeln('NEW TABLE');
{$ENDIF}
  if not Assigned(curtable) then
  begin
    curtable := TEETable.Create;
    curtable.safety := $1234FEED;
    // curtable.rows := Nil;
    curtable.columns := TStringList.Create;
  end
  else
  begin
    for i := 0 to curtable.columns.Count - 1 do
      curtable.columns.Objects[i].Free;
    curtable.columns.Clear;
    if Assigned(curtable.currow) then
      curtable.currow.Free;
  end;
  curtable.pxtop := cury;
  curtable.pagestart := curpage;
  NewRow('default row');
  NewCol('default new table');
  NewCell('default new cell');

end;

procedure Sanity(table: TEETable; msg: string = '');
var
  i: integer;
begin
{$IFDEF DEBUG1}
  writeln('SANITY:', msg);
{$ENDIF}
  try

    if table.safety <> $1234FEED then
      raise Exception.Create('SafetySig fails');
    if Assigned(table.currow) then
      if table.currow.safetysig <> 55554444 then
        raise Exception.Create('currow safety sig failes');

    if Assigned(table.curcol) then
      if table.curcol.safetysig <> $12344321 then
        raise Exception.Create('curcol SafetySig fails');

    if not Assigned(table.columns) then
    begin
      raise Exception.Create('mssing columns');
      exit;
    end;

{$IFDEF DEBUG1}
    writeln('table.columns:', table.columns.ClassName);
{$ENDIF}
    if table.columns.Count = 0 then
      exit;
    // raise Exception.Create('No columnes');

{$IFDEF DEBUG1}
    for i := 0 to table.columns.Count - 1 do
      writeln(i, ' ', table.columns[i]);
{$ENDIF}
  except
    raise Exception.Create('ERROR: doing sanity ');
  end;
end;

function stripsafe(s: ansistring): ansistring;
var
  i: integer;
  ch: ansichar;

begin
  result := '';
  for i := 1 to Length(s) do
  begin
    ch := s[i];
    if ord(ch) > 127 then
      ch := ' ';
    if ord(ch) < 32 then
      break;

    result := result + ch;
  end;

end;

procedure TEEPDFDocument.DrawTableLines;
var
  i: integer;
begin
  i := curtable.currow.pxright;
  { if i < curtable.currow.rightmost then
    i := curtable.currow.rightmost;
  }

  if curtable.pagestart = curpage then
  begin
    QueueLineDraw(curtable.pagestart, curtable.curcol.pxleft, curtable.pxtop,
      curtable.curcol.pxleft, cury, clGreen);
    // rightline
    QueueLineDraw(curtable.pagestart, i, curtable.pxtop, i, cury, clRed);
  end
  else
  begin
    // draw on multiple pages

    QueueLineDraw(curtable.pagestart, curtable.curcol.pxleft, curtable.pxtop,
      curtable.curcol.pxleft, bottommargin, clYellow);
    // rightline
    QueueLineDraw(curtable.pagestart, i, curtable.pxtop, i,
      bottommargin, clBlue);
    QueueLineDraw(curpage, curtable.curcol.pxleft, topmargin,
      curtable.curcol.pxleft, cury, clRed);
    // rightline
    QueueLineDraw(curpage, i, topmargin, i, cury, clBlue);

  end;
end;

// ----------------------------------------------------------------------
procedure TEEPDFDocument.eepdfprocessone(x: TEENode);
var
  i, tempi: integer;
  a: TEEAttrib;
  sub: TEENode;
  stylename, stylefamily: string;
  style: TEEStyle;
  tempwidth: integer;
  s: pdfstring;
  failed: pdfstring;
  oldtabel: TEETableElement;
  oldrow: TEETableRow;
  oldcol: TEETableColumn;
  oldtab: TEETable;
  colindex: integer;
  colsspanned: integer;
  wastable: boolean;
  op: string;
  j: integer;
  hastrailingspace: boolean;
  hasleadingspace: boolean;
  changed: boolean;
  dobold: boolean;
  tempstyle: TEEStyle;
  tempfontsize: integer;
begin

  tempfontsize := fontsize;
  Sanity(curtable);
  try
    op := LowerCase(x.nodetype + '.' + x.nodesubtype);
    if (x.nodetype = '') and (x.nodesubtype = '') then
    begin
      if x.data <> '' then
      begin
{$IFDEF DEBUG1}
        writeln('DATA: "', x.data, '"');
{$ENDIF}
        s := x.data;
        if x.parent.attributes.Count > 0 then
        begin
          a := TEEAttrib(x.parent.attributes.Objects[0]);
          if SameText(a.name, 'TEXT') then
            stylename := a.data;
          tempstyle := GetStyle(stylename);
          if Assigned(tempstyle) then
          begin
            dobold := tempstyle.dobold;
            if not fixedfontsize then
              tempfontsize := tempstyle.fontsize;
          end;
        end;

        tempstyle := GetStyle(curstyle);

        if s = '' then
          hastrailingspace := false
        else
        begin
          hasleadingspace := s[1] = ' ';
          hastrailingspace := s[Length(s)] = ' ';
        end;

        somewords.Clear;
        somewords.LineBreak := ' ';
        somewords.text := x.data;

        // curx := curtable.curcell.pxleft;

        changed := false;
        // tempcurpage := curtable.currow.curpage;
        // cury := curtable.safety .cury;

        for i := 0 to somewords.Count - 1 do
        begin
          s := pdfstring(stripsafe(somewords[i]));
          if s = '%%' then
          begin
            insidereplacements := not insidereplacements;
            s := '';

          end
          else if insidereplacements then
          begin
            failed := s;
            s := replacements.Values[LowerCase(Trim(s))];
            if s = '' then
              replacementserror.Add(failed)

          end else
             // handle case where there are embedded %%'s
             s :=  processreplacements( s );

          if (i < somewords.Count - 1) or hastrailingspace then
            s := s + ' ';
          if hasleadingspace then
            s := ' ' + s;

          try

            if s <> '' then
            begin
              if tempfontsize = 0 then
                tempfontsize := fontsize;

              if dobold then
                canvas.SetFont(font, tempfontsize, [pfsbold])
              else
                canvas.SetFont(font, tempfontsize, []);

              tempwidth := round(canvas.TextWidth(s));
            end;
          except
            raise Exception.Create(format('ERROR processing : %s', [s]));
          end;

          if curx + tempwidth > curtable.curcell.pxright then
          begin
            // wordwrap
            cury := cury - (3 + maxlineheight); // lineheight;

            { if Assigned(curtable.currow) then
              curtable.currow.pxtop := cury;
            }
            curx := curtable.curcell.pxleft;
            // pagination
            if cury < bottommargin then
            begin
              // Addpage;
              inc(tempcurpage);
              cury := topmargin;
            end;
          end;

          if Trim(s) <> '' then
          begin
            // if tempcurpage > 0 then
            // writeln('newpage');

            // do we need a larger line height
            if tempfontsize > maxlineheight then
              maxlineheight := tempfontsize +3 ;

            QueueDrawLater(tempcurpage, curx, cury, s, dobold, tempfontsize);
            if curtable.curcol.rightmost < curx + tempwidth then
              curtable.curcol.rightmost := curx + tempwidth;

            // nextline;  // remove this;
          end;
          // canvas.textout(curx, cury, t);

          curx := curx + tempwidth;
        end;
        somewords.Clear;
        Sanity(curtable, 'post');
      end;
      // save lowest so far
      if curtable.currow.pxbottom > cury then
        curtable.currow.pxbottom := cury;

    end


    // end of

    // ------------------ STYLE ------------------------
    else if SameText(x.nodetype, 'style')
    { and SameText(x.nodesubtype, 'style') }
    then
    begin
      stylename := '';
      stylefamily := '';
      colsspanned := 1; // default

      for i := 0 to x.attributes.Count - 1 do
      begin
        a := TEEAttrib(x.attributes.Objects[i]);
        if SameText(a.name, 'STYLE') then
        begin
          if SameText(a.nametype, 'NAME') then
          begin
            stylename := a.data;
            x.stylename := stylename;
          end

          else if SameText(a.nametype, 'FAMILY') then
            stylefamily := a.data
            // if SameText(a.nametype, 'FONT') ;
          else if SameText(a.nametype, 'number-columns-spanned') then
            colsspanned := StrToIntDef(a.data, 1)
          else if SameText(a.nametype, 'font-family-generic') then
          begin
          end
          else if SameText(a.nametype, 'font-pitch') then
          begin
          end
          else if SameText(a.nametype, 'TEXT-PROPERITES') then
            dobold := false;

        end
        else if SameText(a.name, 'FO') then
        begin
          if SameText(a.nametype, 'font-weight') then
          begin
            if pos('bold', LowerCase(a.data)) > 0 then
              dobold := True
            else
              dobold := false;
            stylename := x.parent.stylename;
            tempstyle := GetStyle(stylename);
            if Assigned(tempstyle) then
              tempstyle.dobold := dobold;

          end
          else if SameText(a.nametype, 'font-size') then
          begin
            s := a.data;
            for tempi := 1 to Length(s) do
              case s[tempi] of
                '.', 'A' .. 'Z', 'a' .. 'z':
                  break;
              end;

            tempi := StrToInt(COpy(a.data, 1, tempi - 1));
            if tempi > 0 then
            begin

              stylename := x.parent.stylename;
              tempstyle := GetStyle(stylename);
              if Assigned(tempstyle) then
                tempstyle.fontsize := tempi;
            end;
          end;
        end;

        if (stylename <> '') and (stylefamily <> '') then
        begin
          style := AddStyle(stylename, stylefamily, x);
          style.columnsspanned := colsspanned;
          style.dobold := dobold;
        end;

      end
    end
    else if SameText(x.nodetype, 'TEXT') then
    begin

      // look at attribute
      for i := 0 to x.attributes.Count - 1 do
      begin
        a := x.attributes.Objects[i] as TEEAttrib;
        if SameText(a.name, 'TEXT') and SameText(a.nametype, 'STYLE-NAME') then
        begin
          curstyle := a.data;
{$IFDEF DEBUG1}
          writeln('STYLE : ', a.data);
{$ENDIF}
        end;
      end;
      if SameText(x.nodesubtype, 'P') then
        nextline(0);

      { erick1
        dec(cury, 15);
        curx := curtable.curcell.pxleft;
      }
    end
    else if SameText(x.nodesubtype, 'TAB') then
    begin
{$IFDEF DEBUG1}
      writeln('TAB');
{$ENDIF}
      curx := curx + CPointsPerInch;
      curx := curx - (curx mod CPointsPerInch);
    end

    // ------------------------ TABLE --------------------
    else if SameText(x.nodetype, 'TABLE') then
    begin
      if SameText(x.nodesubtype, 'TABLE') then
      begin
        curstyle := GetSubAttrSub(x, 'TABLE', 'STYLE-NAME');
        if curstyle = '' then
          raise Exception.Create('Table missing style');

        // curstyle := x.data;
{$IFDEF DEBUG1}
        writeln('TABLETABLE!');
{$ENDIF}
        NewTable;
        // set it to empty

        curtable.curcol.Free;
        curtable.curcol := Nil;

        curtable.columns.Clear;

        dec(lasty, lineheight);;

        curtable.width := ExtractPixelString(GetSubAttr(curstyle, 'STYLE',
          'WIDTH'));
        wastable := True;
      end

      // ---------------------- TABLE-COLUMN---------------
      else if SameText(x.nodesubtype, 'TABLE-COLUMN') then
      begin
        curstyle := GetSubAttrSub(x, 'TABLE', 'STYLE-NAME');
        if curstyle = '' then
          raise Exception.Create('Table missing style');
{$IFDEF DEBUG1}
        writeln('TABLE COLUMN!');
{$ENDIF}
        if not Assigned(curtable) then
          raise Exception.Create('Curtable is missing');

        oldcol := curtable.curcol;

        NewCol(curstyle);

        if Assigned(oldcol) then
        begin
          curtable.curcol.pxleft := oldcol.pxright;
        end
        else
          curtable.curcol.pxleft := leftmargin;

        // curx := curtable.curcol.pxleft;
        // cury := curtable.curcol.pxtop;

        s := GetSubAttr(curstyle, 'STYLE', 'COLUMN-WIDTH');
        style := GetStyle(curstyle);
        colsspanned := style.columnsspanned;
        curtable.curcol.columnsspanned := colsspanned;

        curtable.curcol.width := ExtractPixelString(s);
        if curtable.curcol.width > 0 then
          curtable.curcol.pxright := curtable.curcol.pxleft +
            curtable.curcol.width;
        x.left := curtable.curcol.pxleft;
{$IFDEF DEBUG1}
        writeln('NEW COLUMN START ', curtable.curcol.pxleft, ',',
          curtable.curcol.pxright, ' : ', curtable.columns.Count);
{$ENDIF}
      end
      // ------------------- TABLE-ROW---------------------
      else if SameText(x.nodesubtype, 'TABLE-ROW') then
      begin

        if not curtable.isequalized then
        begin
          // reequalizecolumns;
          curtable.isequalized := True;
        end;
        curstyle := GetSubAttrSub(x, 'TABLE', 'STYLE-NAME');
        if curstyle = '' then
          raise Exception.Create('Table missing style');
{$IFDEF DEBUG1}
        writeln('TABLE ROW!', curstyle);
{$ENDIF}
        oldrow := curtable.currow;

        curtable.cellindex := 0;

        dec(lasty, lineheight);
        NewRow(curstyle);
        cury := lasty;
        curpage := lastpage;
        nextline(0);
        with curtable.currow do
        begin
          pxtop := lasty;
          // start at bottom of so far
          toppage := lastpage;
        end;
        if Assigned(oldrow) then
          oldrow.Free;
        QueueLineDraw(curpage, leftmargin, cury, rightmargin, cury, clBlack);
        DrawTableLines;
      end

      // ----------------- TABLE-CELL --------------------------
      else if SameText(x.nodesubtype, 'TABLE-CELL') then
      begin
        curstyle := GetSubAttrSub(x, 'TABLE', 'STYLE-NAME');
        s := GetSubAttrSub(x, 'TABLE', 'NUMBER-COLUMNS-SPANNED');
        colsspanned := StrToIntDef(s, 1);

        if curstyle = '' then
          // raise Exception.Create('Table missing style');
{$IFDEF DEBUG1}
          writeln('TABLE CELL!');
{$ENDIF}
        oldtabel := curtable.curcell;
        // do we need to record the bottom

        NewCell(curstyle);

        colindex := curtable.cellindex;

{$IFDEF DEBUG1}
        if Assigned(curtable.curcol) then
          writeln('using table col ', curtable.curcol.name);
{$ENDIF}
        if colindex > curtable.columns.Count - 1 then
          raise Exception.Create(format('Invalid column index %d >= %d',
            [colindex, curtable.columns.Count]));

        curtable.curcol := curtable.columns.Objects[colindex] as TEETableColumn;

        curtable.curcell.pxleft := leftmargin;

        // oldtabel.Free;       ERICK
        if colindex > 0 then
        begin
          oldcol := curtable.columns.Objects[colindex - 1] as TEETableColumn;
          curtable.curcell.pxleft := oldcol.pxright + 20;
        end;
        s := GetSubAttr(curstyle, 'STYLE', 'COLUMN-WIDTH');
        style := GetStyle(curstyle);

        curtable.curcell.width := curtable.curcol.width;

        if curtable.curcol.width > 0 then
          curtable.curcell.pxright := curtable.curcell.pxleft +
            curtable.curcell.width;
        FixCurrentCellWidth(colsspanned);

        curx := curtable.curcol.pxleft;
        cury := curtable.currow.pxtop;
        tempcurpage := curpage;

        // must be last thing we do
        inc(curtable.cellindex, colsspanned); // for next time

      end
      else
      begin
{$IFDEF DEBUG1}
        write('Table: ', x.nodesubtype);
{$ENDIF}
      end;
      // writeln(a.data);
    end;
  except
    Exception.Create(format('ERROR: processing %s : %s', [x.nodetype, op]));
    raise;
  end;

  for i := 0 to x.subnodes.Count - 1 do
  begin
    sub := x.subnodes.Objects[i] as TEENode;
    if not Assigned(sub) then
      raise Exception.Create('Missing node');

    eepdfprocessone(sub);
  end;

{$IFDEF DEBUG1}
  if op <> '.' then
    writeln('op: ', op);
{$ENDIF}
  if op = 'table.table-cell' then
  begin
  end;
  if op = 'table.table-row' then
  begin
  end;
  if op = 'table.table-column' then
  begin
    if Assigned(curtable.curcol) then
    begin
      DrawTableLines;
    end;
  end;
  if op = 'table.table' then
  begin
    NewTable;
    if lasty <> 0 then
    begin
      cury := lasty;
      curpage := lastpage;
    end;
    nextline(0);
  end;

end;

// ----------------------------------------------------------------------
function TEEPDFDocument.GeneratePDF: string;
var
  LFormatWidth, LWidth, LHeight, LI: integer;
  LPages: TList;
  LScale: Single;
  LMarginX, LMarginY, LPointsWidth, LPointsHeight, LMarginBottom: Single;
  PageText: string;

  eenode: TEENode;
  x, y: integer;
begin
  // ForceJPEGCompression := 80;
  // MarginLeft := 1;

  // LPointsWidth := DefaultPageWidth - Centimeters2Points(MarginLeft + MarginRight);

  // LFormatWidth := Inches2Points(8.0);

  // LPointsHeight := (DefaultPageHeight - Centimeters2Points(MarginTop +    MarginBottom)) / LScale;
  // LMarginX := Centimeters2Points(MarginLeft);
  // LMarginY := -Centimeters2Points(MarginTop);
  // LMarginBottom := Centimeters2Points(MarginBottom);

  ScreenLogPixels := CPointsPerInch;

  // start at first page
  LI := 0;
  NewDoc;

  leftmargin := Trunc(Inches2Points(0.5));
  rightmargin := Trunc(Inches2Points(8.5 - 0.5));
  topmargin := Trunc(Inches2Points(11 - 0.5));
  bottommargin := Trunc(Inches2Points(0.5));

  eenode := top;
  curx := 0;
  cury := 0;
  lasty := 0;

  NewTable;
  {
    NewCol('default special');
    NewRow('default special');
    NewCell('default special');
  }
  with curtable.curcell do
  begin
    pxleft := leftmargin;
    pxright := rightmargin;
    pxtop := topmargin;
    pxbottom := topmargin;
    // bottommargin;
  end;

  NewPage;

  repeat
    if cury < bottommargin then
    begin

      // lpage := Addpage;
      curx := leftmargin;
      cury := topmargin;
    end;

    // LPage := TMetafile(LPages[LI]);
    with canvas do
    begin
      {
        GSave;
        Rectangle(LMarginX, LMarginBottom, LPointsWidth, LPointsHeight);
        Clip; // THtmlView may print out of the margins ;)
        RenderMetaFile(LPage, LScale, LMarginX, LMarginY);
        GRestore;
      }
      // do an element
      // if node then

      if eenode = Nil then
        break;

      // PageText := 'Page %d/%d';
      // PageText := Format(PageText, [LI + 1,5]);

      eepdfprocessone(eenode);

      // eenode := GetNextNode;
    end;
  until True;
  NowDrawQueue;
end;

procedure TEEPDFDocument.LoadFromODT(filename: string);
var
  zr: TZipRead;
  ss: TStringStream;
  i: integer;
  s: string;
const
  contentxml = 'content.xml';
begin
  try
    s := '';
    ss := TStringStream.Create;
    zr := TZipRead.Create(filename);
    i := zr.NameToIndex(contentxml);
    if i < 0 then
      raise Exception.Create('content.xml not found in file');
    zr.unzip(i, ss);
    s := ss.DataString;
    top := ParseEEXML(s);
  finally
    zr.Free;
    ss.Free;

  end;

end;

{ DELPHI version - bad because it overwrites content.xml in local dir
  procedure TEEPDFDocument.LoadFromODT2(filename: string);
  var
  i: integer;
  zip: TZipFile;
  const
  contentxml = 'content.xml';
  begin

  if fileexists(contentxml) then
  DeleteFile(contentxml);

  zip := TZipFile.Create;
  zip.open(filename, zmRead);
  for i := 0 to zip.FileCount - 1 do
  begin
  if zip.filename[i] = contentxml then
  begin
  zip.Extract(contentxml, '', false);
  LoadFromXMLFile(contentxml);
  zip.Free;
  exit;
  end;
  end;
  zip.Free;
  raise Exception.Create('content.xml not found in zip');
  end;
}

function TEEPDFDocument.processreplacements(s: string): string;
var
  start, fin: integer;
  name: string;
  replace: string;
const
  head: string = '%%';
begin
  // if it was started earlier
  if insideSpanreplace <> '' then
  s := insideSpanReplace + s;
  // reset it for now, that we've copied it
  insidespanreplace := '';
  repeat
    start := pos(head, s);
    if start > 0 then
    begin
      // found one
      name := COpy(s, start + 2);
      fin := pos('%%',name );
      if fin < 1 then begin
        // it will scroll onto next string
        insideSpanReplace := name;
        // temporary result
        s := copy( s, 1, start - 1 );
        break;
      end;
      SetLength(name, fin - 1);

      replace := replacements.Values[LowerCase(name)];
      if replace = '' then replacementserror.Add( name );

      s := COpy(s, 1, start - 1) + replace + COpy(s, start + fin + 4);
    end;
  until start < 1;
  result := s;
end;

end.
